/* ...............................................................

	WindowColors
	Copyright 1997-8 Steve Klingsporn <moofie@pobox.com>
	Based on WindowShade by Marco Nelissen <marcone@xs4all.nl>
	
		File:	WindowView.cpp
	
	Contains:	Implementation of a BView subclass that does
				its damndest to look like a BWindow, using
				colors from a rgb_color array stored in its
				parent view.
	
	   Notes:	None.
	   
   ............................................................... */

#ifndef _WINDOW_VIEW_H
#include "WindowView.h"
#endif

/* ...............................................................
	String constants		
   ............................................................... */

const char		*ACTIVE_LABEL = "Active";
const char		*INACTIVE_LABEL = "Inactive";
const char		*BORDER_LABEL = "Border";
const char		*MINIMIZED_LABEL = "Minimized";


/* ...............................................................	
	WindowView Constructor
	Sets the view color to transparent (we draw all the bits in
	the view which need updating).
   ............................................................... */

WindowView::WindowView(BRect frame, const char *name, 
					   window_type look)
		    :BView(frame, name, B_FOLLOW_NONE,
		    					B_WILL_DRAW |
		    					B_NAVIGABLE_JUMP),
		     _look(look),
		     _active(false)
{
	SetViewColor(RGB_COLOR_216);
}


/* ...............................................................	
	WindowView::ParentPanel()
	Returns the parent view with the right type.
   ............................................................... */

WindowColorsPanel *WindowView::ParentPanel()
{
	return ((WindowColorsPanel *)Parent());
}


/* ...............................................................	
	WindowView::MouseDown()
	Called to be friendly and try to focus the color on where the
	user is clicking.  We'll guess that if the user clicks on the
	tab, they want to change the tab color, and that if she has
	the option down, she wants to set the inactive tab color.
	If she clicks on the frame, since it's so thin an area, maybe
	she wants to change the frame color (same deal with the option
	key there, too).  Look in MessageReceived() for the same deal,
	only with color dragging.
   ............................................................... */

void WindowView::MouseDown(BPoint where)
{
	WindowColorsPanel *parent = ParentPanel();
	bool inactive = (parent->ActiveWindowView() != this);
	
	/* Activation */
	if (inactive || _tabBounds.Contains(where))
	{
		if (! _active && modifiers() & B_OPTION_KEY)
			ChildAt(1)->MakeFocus(true);	/* Inactive divot */
		else
			ChildAt(0)->MakeFocus(true);	/* Active divot */
		if (inactive)
			return;
	}
	
	if (CLOSE_BOX_BOUNDS.Contains(where))
	{
		uint32 buttons;
		GetMouse(&where, &buttons);
		BBitmap *bitmap;

		while (buttons)
		{			
			if (CLOSE_BOX_BOUNDS.Contains(where))
			{
				bitmap = parent->Bitmap(CLOSE_BOX_ON_BITMAP);
				DrawBitmap(bitmap, bitmap->Bounds(),
						   CLOSE_BOX_BOUNDS);
			}
			else
			{
				bitmap = parent->Bitmap(CLOSE_BOX_OFF_BITMAP);
				DrawBitmap(bitmap, bitmap->Bounds(),
						   CLOSE_BOX_BOUNDS);
			}
			snooze(20000);
			GetMouse(&where, &buttons);
		}
		bitmap = parent->Bitmap(CLOSE_BOX_OFF_BITMAP);
		DrawBitmap(bitmap, bitmap->Bounds(),
				   		CLOSE_BOX_BOUNDS);
	}
	
	BRect zoomBoxBounds = CLOSE_BOX_BOUNDS;
	zoomBoxBounds.OffsetTo(_tabBounds.right - 13 - 4, 4);
						  
	if (zoomBoxBounds.Contains(where))
	{
		uint32 buttons;
		GetMouse(&where, &buttons);
		BBitmap *bitmap;

		while (buttons)
		{			
			if (zoomBoxBounds.Contains(where))
			{
				bitmap = parent->Bitmap(ZOOM_BOX_ON_BITMAP);
				DrawBitmap(bitmap, bitmap->Bounds(),
						   zoomBoxBounds);
			}
			else
			{
				bitmap = parent->Bitmap(ZOOM_BOX_OFF_BITMAP);
				DrawBitmap(bitmap, bitmap->Bounds(),
						   zoomBoxBounds);
			}
			snooze(20000);
			GetMouse(&where, &buttons);
		}
		bitmap = parent->Bitmap(ZOOM_BOX_OFF_BITMAP);
		DrawBitmap(bitmap, bitmap->Bounds(),
				   		zoomBoxBounds);
	}
}	


/* ...............................................................	
	WindowView::AttachedToWindow()
	Sets up usage-specific subviews and all sorts of other nasty
	stuff.
   ............................................................... */

void WindowView::AttachedToWindow()
{
	/* Create a font to size with */
	BFont font(be_bold_font);
	if (font.Size() > 14)
		font.SetSize(14);

	/* Set up some common variables for the switch */	 	
	BRect divotFrame = DIVOT_LABEL_FRAME;
	float spacer = divotFrame.Height() + 4;
	ColorLabel *label;
	
	switch (_look)
	{
		case B_TITLED_WINDOW:
		{
			/* Set the tab bounds */
			_tabBounds.Set(0, 0, font.StringWidth(Name()) + 68,
												  TAB_HEIGHT);
			/* Create 5 of the 6 children, based on
			   their positions in the window_colors
			   array */			
			for (uint8 index = ACTIVE_TAB_BASE_COLOR; 
				 index < INACTIVE_FRAME_SHADOW_COLOR; 
				 index += 3)
			{
				/* Determine the label for the
				   ColorLabel */
				const char *title;
				switch (index)
				{
					case ACTIVE_TAB_BASE_COLOR:
					case ACTIVE_FRAME_BASE_COLOR:
						title = ACTIVE_LABEL;
						break;
					
					case MINIMIZED_TAB_BASE_COLOR:
						title = MINIMIZED_LABEL;
						break;
						
					case INACTIVE_TAB_BASE_COLOR:
					case INACTIVE_FRAME_BASE_COLOR:
						title = INACTIVE_LABEL;
						break;
				}
				
				/* Create the ColorLabel, add it, and adjust
				   the sizing frame */
				label = new ColorLabel(divotFrame,
											title,
											(window_color)index);
				AddChild(label);
				if (index == MINIMIZED_TAB_BASE_COLOR)
					divotFrame.OffsetTo(FRAME_DIVOT_ORIGIN);
				else
					divotFrame.OffsetBy(0, spacer);
			}
			
			/* Add the "Border" ColorLabel */
			label = new ColorLabel(divotFrame,
										BORDER_LABEL,
										BORDER_COLOR);
			AddChild(label);
		}
		break;
		
		case B_MODAL_WINDOW:
		{
			_tabBounds.Set(0, 0, 5, 0);
			
			BRect sizer = DIVOT_LABEL_FRAME;
			float spacer = sizer.Height() + 4;
	
			/* Active modal frame divot label */
			sizer.OffsetTo(MODAL_FRAME_DIVOT_ORIGIN);
			label = new ColorLabel(sizer,
										ACTIVE_LABEL,
										ACTIVE_MODAL_FRAME_BASE_COLOR);
			AddChild(label);
			
			/* Inactive modal frame divot label */
			sizer.OffsetBy(0, spacer);
			label = new ColorLabel(sizer,
										INACTIVE_LABEL,
										INACTIVE_MODAL_FRAME_BASE_COLOR);
			AddChild(label);
		}
		break;
	}
}


/* ...............................................................	
	WindowView::Update()
	Called to update the WindowView and child ColorLabels
	when the color model changes from underneath us.
   ............................................................... */

void WindowView::Update()
{
	Draw(Bounds());
	uint8 i;
	
	switch (_look)
	{
		case B_TITLED_WINDOW:
			for (i = 0; i < 6; i++)
				ChildAt(i)->Draw(ChildAt(i)->Bounds());
			break;
			
		case B_MODAL_WINDOW:
			for (i = 0; i < 2; i++)
				ChildAt(i)->Draw(ChildAt(i)->Bounds());
			break;
	}
}


/* ...............................................................	
	WindowView::Draw()
	Draws this window view, in the correct colors, exactly likee
	Be windows look, with an additional subtle drop shadow.
   ............................................................... */

void WindowView::Draw(BRect updateRect)
{
	BRect 				bounds = Bounds();
	WindowColorsPanel	*parent = ParentPanel();
	
	/* Subtract 2 from the right and bottom edges
	   of the bounds rectangle to accomodate the
	   subtle drop shadow. */
	bounds.right -= 2;
	bounds.bottom -= 2;
	
	/* The colors we will use to paint our window model. */
	rgb_color borderColor, borderShadowColor;
	rgb_color frameColor, frameHiliteColor, frameShadowColor;
	rgb_color tabColor, tabHiliteColor, tabShadowColor;

	/* The border color is always the same, and the window
	   draws a darker line on the right and bottom border edges,
	   so we need to calculate (but not store) this color too. */
	   
	borderColor = parent->Color(BORDER_COLOR);
	borderShadowColor = parent->Color(BORDER_SHADOW_COLOR);
	
	/* Determine the proper colors to draw in based on our state*/
	if (_active)
	{
		/* Active normal window */
		tabHiliteColor = 
			parent->Color(ACTIVE_TAB_HILITE_COLOR);
		tabColor = 
			parent->Color(ACTIVE_TAB_BASE_COLOR);
		tabShadowColor = 
			parent->Color(ACTIVE_TAB_SHADOW_COLOR);
		
		if (_look != B_MODAL_WINDOW)
		{
			/* Active normal window */
			frameHiliteColor = 
				parent->Color(ACTIVE_FRAME_HILITE_COLOR);
			frameColor = 
				parent->Color(ACTIVE_FRAME_BASE_COLOR);
			frameShadowColor = 
				parent->Color(ACTIVE_FRAME_SHADOW_COLOR);
		}
		else
		{
			/* Active modal window */
			frameHiliteColor = 
				parent->Color(ACTIVE_MODAL_FRAME_HILITE_COLOR);
			frameColor = 
				parent->Color(ACTIVE_MODAL_FRAME_BASE_COLOR);
			frameShadowColor = 
				parent->Color(ACTIVE_MODAL_FRAME_SHADOW_COLOR);
		}
	}
	else
	{
		/* Inactive normal window */
		tabHiliteColor = 
			parent->Color(INACTIVE_TAB_HILITE_COLOR);
		tabColor = 
			parent->Color(INACTIVE_TAB_BASE_COLOR);
		tabShadowColor = 
			parent->Color(INACTIVE_TAB_SHADOW_COLOR);
		
		if (_look != B_MODAL_WINDOW)
		{
			/* Inactive normal window */
			frameHiliteColor = 
				parent->Color(INACTIVE_FRAME_HILITE_COLOR);
			frameColor = 
				parent->Color(INACTIVE_FRAME_BASE_COLOR);
			frameShadowColor = 
				parent->Color(INACTIVE_FRAME_SHADOW_COLOR);
		}
		else
		{
			/* Inactive modal window */
			frameHiliteColor = 
				parent->Color(INACTIVE_MODAL_FRAME_HILITE_COLOR);
			frameColor = 
				parent->Color(INACTIVE_MODAL_FRAME_BASE_COLOR);
			frameShadowColor = 
				parent->Color(INACTIVE_MODAL_FRAME_SHADOW_COLOR);
		}
	}
	
	/* Line arrays are fast, and we'll try to cram as many
	   drawing calls as we can into one first */
	BeginLineArray(26);
	
	/* Draw the window frame, its look depending on which
	   look we have selected for this view. */
	if (_look != B_MODAL_WINDOW)
	{
		/* Draw the regular window frame */
		AddLine(BPoint(0, 0), 
				BPoint(0, bounds.bottom - 1), 
				borderColor);
		AddLine(BPoint(0, bounds.bottom), 
				BPoint(bounds.right, bounds.bottom),
				borderShadowColor);
		AddLine(BPoint(bounds.right, bounds.bottom - 1), 
				BPoint(bounds.right, _tabBounds.bottom), 
				borderShadowColor);
		AddLine(BPoint(bounds.right - 1, _tabBounds.bottom), 
				BPoint(_tabBounds.right + 1, _tabBounds.bottom), 
				borderColor);
		AddLine(BPoint(_tabBounds.right, _tabBounds.bottom), 
				BPoint(_tabBounds.right, 0), 
				borderShadowColor);
		AddLine(BPoint(_tabBounds.right - 1, 0), 
				BPoint(1, 0), 
				borderColor);
	}
	else
	{
		/* Draw the modal window frame */
		AddLine(BPoint(0, 0), 
				BPoint(0, bounds.bottom), 
				borderColor);
		AddLine(BPoint(0, bounds.bottom), 
				BPoint(bounds.right, bounds.bottom),
				borderShadowColor);
		AddLine(BPoint(bounds.right, bounds.bottom), 
				BPoint(bounds.right, 0), 
				borderShadowColor);
		AddLine(BPoint(bounds.right - 1, 0), 
				BPoint(0, 0), 
				borderColor);
	}
	
	/* Draw the window frame with the right colors */
	AddLine(BPoint(3, _tabBounds.bottom + 1), 
			BPoint(_tabBounds.right, _tabBounds.bottom + 1), 
			frameColor);
	AddLine(BPoint(3, _tabBounds.bottom + 2), 
			BPoint(bounds.right - 3, _tabBounds.bottom + 2), 
			frameColor);
	AddLine(BPoint(2, _tabBounds.bottom + 1), 
			BPoint(2, bounds.bottom - 2), 
			frameColor);
	AddLine(BPoint(3, bounds.bottom - 2), 
			BPoint(bounds.right - 2, bounds.bottom - 2), 
			frameColor);
	AddLine(BPoint(bounds.right - 2, bounds.bottom - 3), 
			BPoint(bounds.right - 2, _tabBounds.bottom + 2), 
			frameColor);
	AddLine(BPoint(3, _tabBounds.bottom + 3), 
			BPoint(3, bounds.bottom - 3),
			frameShadowColor);
	AddLine(BPoint(4, _tabBounds.bottom + 3), 
			BPoint(bounds.right - 4, _tabBounds.bottom + 3), 
			frameShadowColor);
	AddLine(BPoint(bounds.right - 1, _tabBounds.bottom + 1), 
			BPoint(bounds.right - 1, bounds.bottom - 1), 
			frameShadowColor);
	AddLine(BPoint(bounds.right - 1, bounds.bottom - 1), 
			BPoint(1, bounds.bottom - 1), 
			frameShadowColor);
	AddLine(BPoint(1, _tabBounds.bottom + 1), 
			BPoint(1, bounds.bottom - 2), 
			frameHiliteColor);
			
	if (_look != B_MODAL_WINDOW)
	{
		AddLine(BPoint(_tabBounds.right + 1, 
				_tabBounds.bottom + 1), 
				BPoint(bounds.right - 2, 
				_tabBounds.bottom + 1), 
				frameHiliteColor);
	}
	else
	{
		AddLine(BPoint(1, 1), 
				BPoint(bounds.right - 2, 1), 
				frameHiliteColor);
	}
	AddLine(BPoint(bounds.right - 3, _tabBounds.bottom + 3), 
			BPoint(bounds.right - 3, bounds.bottom - 3), 
			frameHiliteColor);
	AddLine(BPoint(bounds.right - 4, bounds.bottom - 3), 
			BPoint(4, bounds.bottom - 3), 
			frameHiliteColor);
	
	/* Draw the tab hilites and shadow lines
	   (a bit out-of-order, but we sneak them into
	   the single line array. */
	if (_look != B_MODAL_WINDOW)
	{
		AddLine(BPoint(_tabBounds.right - 1, 0 + 1), 
				BPoint(_tabBounds.right - 1, _tabBounds.bottom), 
				tabShadowColor);
		AddLine(BPoint(_tabBounds.right - 2, 0 + 1), 
				BPoint(1, 0 + 1), 
				tabHiliteColor);
		AddLine(BPoint(1, 0 + 2), 
				BPoint(1, _tabBounds.bottom), 
				tabHiliteColor);
	}
	
	/* Draw the inside "panel" in the window in the
	   standard gray base color, and touch up some
	   places we need to update if necessary. */		
	SetHighColor(RGB_COLOR_216);
	
	BRect panelBounds = BRect(5, _tabBounds.bottom + 5,
						      bounds.right - 5, bounds.bottom - 5);
						      
	if (updateRect.Intersects(panelBounds))
	{
		panelBounds.InsetBy(1, 1);
		FillRect(panelBounds & updateRect);
		AddLine(BPoint(bounds.right - 5, _tabBounds.bottom + 5), 
				BPoint(bounds.right - 5, bounds.bottom - 5), 
				RGB_COLOR_192);
		AddLine(BPoint(bounds.right - 6, bounds.bottom - 5), 
				BPoint(5, bounds.bottom - 5), 
				RGB_COLOR_192);
		AddLine(BPoint(5, bounds.bottom - 6), 
				BPoint(5, _tabBounds.bottom + 5), 
				WHITE_COLOR);
		AddLine(BPoint(6, _tabBounds.bottom + 5), 
				BPoint(bounds.right - 5, _tabBounds.bottom + 5), 
				WHITE_COLOR);
	}

	EndLineArray();

	/* Draw a subtle drop-shadow from the window */		
	SetHighColor(200, 200, 200, 255);
	FillRect(BRect(bounds.right + 1, 
				   _tabBounds.bottom + 4, 
			 	   bounds.right + 2, bounds.bottom + 2));
	FillRect(BRect(4, bounds.bottom + 1, bounds.right + 2, 
				   bounds.bottom + 2));
	FillRect(BRect(_tabBounds.right + 1, 
				   bounds.top + 4, 
				   _tabBounds.right + 2, 
				   _tabBounds.bottom - 1));
	
	/* Draw in the interior rectangle with the border color */
	bounds.InsetBy(4, 4);
	bounds.top += _tabBounds.bottom;	
	SetHighColor(borderColor);
	StrokeRect(bounds);
	
	/* If we're not a modal window, draw the tab */
	if (_look != B_MODAL_WINDOW)
	{
		bounds.Set(2, 2, _tabBounds.right - 2, _tabBounds.bottom);
		SetLowColor(tabColor);
		FillRect(bounds, B_SOLID_LOW);
		
		/* If the window is inactive, draw the thin
		   line that Be windows appear to have */
		if (! _active)
		{
			SetHighColor(201, 201, 201);
			StrokeLine(BPoint(1, _tabBounds.bottom),
					   BPoint(_tabBounds.right - 1,
					  		  _tabBounds.bottom));
		}
		
		/* Draw the window title in the tab if necessary */
		if (updateRect.Intersects(_tabBounds))
		{
			SetHighColor(BLACK_COLOR);
			SetFont(be_bold_font);
			if (be_bold_font->Size() > 14)
				SetFontSize(14);
			MovePenTo(35, 16);
			DrawString(Name());
		}
		
		/* Draw the close and zoom boxes, if necessary */
		if (_active)
		{
			WindowColorsPanel *parent = ParentPanel();
			BBitmap *bitmap = parent->Bitmap
							  (CLOSE_BOX_OFF_BITMAP);
			if (bitmap != NULL && updateRect.Intersects(
								  CLOSE_BOX_BOUNDS))
				DrawBitmap(bitmap, bitmap->Bounds(),
								   CLOSE_BOX_BOUNDS);
								   
			bitmap = parent->Bitmap(ZOOM_BOX_OFF_BITMAP);
			BRect zoomBoxBounds = CLOSE_BOX_BOUNDS;
			zoomBoxBounds.OffsetTo(_tabBounds.right - 13 - 4, 4);
			if (bitmap != NULL && updateRect.Intersects(
									zoomBoxBounds))
				DrawBitmap(bitmap, bitmap->Bounds(),
									zoomBoxBounds);
		}
	}
}


/* ...............................................................	
	MessageReceived
	Overridden to support special color drops (on the title tab,
	or the minimized title tab).
   ............................................................... */

void WindowView::MessageReceived(BMessage *message)
{
	if (message->WasDropped())
	{
		BPoint dropPoint = message->DropPoint();
		ConvertFromScreen(&dropPoint);
		if (_tabBounds.Contains(dropPoint))
		{
			rgb_color *color;
			ssize_t	  size;
			
			if (message->FindData("RGBColor", B_RGB_COLOR_TYPE,
							  	  &color, &size) == B_NO_ERROR)
			{
				if (! IsActive() && (modifiers() & B_OPTION_KEY))
					((ColorLabel *)ChildAt(1))->SetColor(*color);
				else
					((ColorLabel *)ChildAt(0))->SetColor(*color);
			}
		}
	}
	else
		BView::MessageReceived(message);
}


/* ...............................................................
	WindowView::IsActive()
	Returns whether or not this window view is active.
   ............................................................... */

bool WindowView::IsActive()
{
	return _active;
}


/* ...............................................................
	WindowView::Activate()
	Activates or deactivates this window view.
   ............................................................... */

void WindowView::Activate(bool flag)
{
	if (_active != flag)
	{
		_active = flag;
		if (! IsHidden())
			Draw(_tabBounds);
	}
}